/**
 * @author Department of Data Science and Knowledge Engineering (DKE)
 * @version 2022.0
 */

import java.util.Arrays;
import java.util.Scanner;

/**
 * This class includes the methods to support the search of a solution.
 */
public class Search
{
	public static int horizontalGridSize;
	public static int verticalGridSize;
	public static final int EMPTY_SPACE_FIELD = -1;
	public static final int EMPTY_SPACE_PENTOMINO = 0;

	public static final char BACK_TRACK_ALGORITHM_ID = 'a';
	public static final char BRUTE_FORCE_ALGORITHM_ID = 'b';

	public static char[] input;

	public static UI ui;

	/**
	 * Helper function which starts by default and gets the selected algorithm solver that the user wants to execute.
	 * @param args command line arguments
	 */
	public static void main(String[] args) {
		Scanner scanner = new Scanner(System.in);
		char selectedAlgorithm;

		do {
			selectedAlgorithm = getUserAlgorithmInput(scanner);
		} while (selectedAlgorithm != BRUTE_FORCE_ALGORITHM_ID && selectedAlgorithm != BACK_TRACK_ALGORITHM_ID);

		startPentominoSolver(scanner, selectedAlgorithm);
		scanner.close();
	}

	/**
	 * Helper function which starts the pentomino solver algorithm.
	 * @param scanner scanner to read user input
	 * @param selectedAlgorithm character representing the selected algorithm
	 */
	public static void startPentominoSolver(Scanner scanner, char selectedAlgorithm) {
		System.out.print("Enter horizontal grid size: ");
		horizontalGridSize = scanner.nextInt();
		System.out.print("Enter vertical grid size: ");
		verticalGridSize = scanner.nextInt();
		scanner.nextLine(); // Consume the newline character

		boolean validInput = false;
		while (!validInput) {
			System.out.print("Enter the pentominoes (characters without spaces): ");
			String pentominoes = scanner.nextLine();
			input = pentominoes.toCharArray();
			validInput = validatePentominoes(input);

			if (!validInput) {
				System.out.println("Invalid pentominoes input. Please enter valid characters.");
			}
		}

		ui = new UI(horizontalGridSize, verticalGridSize, 50);

		boolean solutionFound = search(selectedAlgorithm);

		System.out.println(solutionFound ? "Solution found!" : "No solution found.");
	}

	/**
	 * Helper function to get the selected algorithm from the user
	 * @param scanner scanner to read user input
	 * @return character representing the selected algorithm
	 */
	private static char getUserAlgorithmInput(Scanner scanner) {
		System.out.println("Select the algorithm you want to use:");
		System.out.println("a - BackTrack Algorithm (Description of Algorithm A)");
		System.out.println("b - Brute Force Algorithm (Description of Algorithm B)");
		System.out.print("Enter your choice (a/b): ");
		return scanner.nextLine().trim().toLowerCase().charAt(0);
	}

	/**
	 * Helper function to validate the pentominoes input from the user (only valid characters)
	 * @param pentominoes array of characters representing the pentominoes
	 * @return true if the input is valid, false otherwise
	 */
	private static boolean validatePentominoes(char[] pentominoes) {
		for (char pentomino : pentominoes) {
			int character = characterToID(Character.toUpperCase(pentomino));
			if (character == EMPTY_SPACE_FIELD) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Helper function which starts a basic search algorithm with the user selection
	 * @param selectedAlgorithm character representing the selected algorithm
	 * @return true if a solution is found, false otherwise
	 */
	public static boolean search(char selectedAlgorithm) {
		int[][] field = new int[horizontalGridSize][verticalGridSize];
		cleanBoard(field);

		if (horizontalGridSize * verticalGridSize % 5 != 0) {
			System.out.println("Not possible to find a solution");
			return false;
		}

		if (selectedAlgorithm == 'a') {
			return backTrackSearch(field, 0);
		} else if (selectedAlgorithm == 'b') {
			return bruteForceSearch(field, 0);
		}

		return false;
	}

	/**
	 * Get as input the character representation of a pentomino and translate it into its corresponding numerical value (ID)
	 * @param character a character representation of a pentomino
	 * @return	the corresponding ID (numerical value)
	 */
	private static int characterToID(char character) {
		int pentID = EMPTY_SPACE_FIELD;
		switch (character) {
			case 'X' -> pentID = 0;
			case 'I' -> pentID = 1;
			case 'Z' -> pentID = 2;
			case 'T' -> pentID = 3;
			case 'U' -> pentID = 4;
			case 'V' -> pentID = 5;
			case 'W' -> pentID = 6;
			case 'Y' -> pentID = 7;
			case 'L' -> pentID = 8;
			case 'P' -> pentID = 9;
			case 'N' -> pentID = 10;
			case 'F' -> pentID = 11;
		}
		return pentID;
	}

	/**
	 * Helper function to check if a piece can be placed on the board
	 * @param field the board
	 * @param pieceToPlace the piece to be placed
	 * @param x x position of the piece
	 * @param y y position of the piece
	 * @return true if the piece can be placed, false otherwise
	 */
	private static boolean canPlacePiece(int[][] field, int[][] pieceToPlace, int x, int y) {
		for (int i = 0; i < pieceToPlace.length; i++) {
			for (int j = 0; j < pieceToPlace[i].length; j++) {
				// X and Y are relative to the piece not the field here!
				if (pieceToPlace[i][j] == EMPTY_SPACE_PENTOMINO) {
					// We are not interested in these i and j values, they are empty spaces and don't provide collisions
					continue;
				}

				// The new values of X and Y (relative to the field) that will be filled by the piece
				int newX = x + i;
				int newY = y + j;

				// Check for out of bound pieces
				if ((newX) >= horizontalGridSize || (newY) >= verticalGridSize ) {
					return false;
				}

				// Check for overlap between piece elements and already placed pieces
				if (field[newX][newY] != EMPTY_SPACE_FIELD) {
					// there is an overlap
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Helper function to clean the board (fill it with empty spaces)
	 * @param field the board
	 */
	private static void cleanBoard(int[][] field) {
		//Empty board again to find a solution
		for (int i = 0; i < field.length; i++) {
			Arrays.fill(field[i], EMPTY_SPACE_FIELD);
		}
	}

	/**
	 * Adds a pentomino to the position on the field (overriding current board at that position)
	 * @param field a matrix representing the board to be fulfilled with pentominoes
	 * @param piece a matrix representing the pentomino to be placed in the board
	 * @param pieceID ID of the relevant pentomino
	 * @param x x position of the pentomino
	 * @param y y position of the pentomino
	 */
	public static void addPiece(int[][] field, int[][] piece, int pieceID, int x, int y) {
		runPiece(field, piece, pieceID, x, y);
	}

	/**
	 * Helper function to add a pentomino to the position on the field (overriding current board at that position)
	 * @param field a matrix representing the board to be fulfilled with pentominoes
	 * @param piece a matrix representing the pentomino to be placed in the board
	 * @param pieceID ID of the relevant pentomino
	 * @param x x position of the pentomino
	 * @param y y position of the pentomino
	 */
	private static void runPiece(int[][] field, int[][] piece, int pieceID, int x, int y) {
		for (int i = 0; i < piece.length; i++) {
			for (int j = 0; j < piece[i].length; j++) {
				if (piece[i][j] != EMPTY_SPACE_PENTOMINO) {
					// Add the ID of the pentomino to the board if the pentomino occupies this square
					field[x + i][y + j] = pieceID;
				}
			}
		}
	}

	/**
	 * Removes a pentomino from the position on the field (overriding current board at that position)
	 * @param board a matrix representing the board to be fulfilled with pentominoes
	 * @param piece a matrix representing the pentomino to be placed in the board
	 * @param x x position of the pentomino
	 * @param y y position of the pentomino
	 */
	public static void removePiece(int[][] board, int[][] piece, int x, int y) {
		runPiece(board, piece, EMPTY_SPACE_FIELD, x, y);
	}

	/**
	 * Brute force search algorithm. It checks all the possible combinations of pentominoes and positions to find a solution.
	 * @param field a matrix representing the board to be fulfilled with pentominoes
	 * @param index places pentomino number "index" on the board (if possible)
	 * Calls itself recursively with the next index (field, 1), (field, 2), etc. - until done
	 */

	private static boolean bruteForceSearch(int[][] field, int index) {
		if (index == input.length) {
			// All pentominoes placed, a solution is found
			ui.setState(field);
			return true;
		}

		// Retrieve the current pentomino character from the input array
		char piece = input[index];
		int pentID = characterToID(piece);

		// Iterate through *all possible* mutations of the current pentomino piece
		for (int mutation = 0; mutation < PentominoDatabase.data[pentID].length; mutation++) {
			int[][] pieceToPlace = PentominoDatabase.data[pentID][mutation];

			// Place the current pentomino piece at *all possible* positions on the grid
			for (int x = 0; x <= horizontalGridSize - pieceToPlace.length; x++) {
				for (int y = 0; y <= verticalGridSize - pieceToPlace[0].length; y++) {

					if (canPlacePiece(field, pieceToPlace, x, y)) {

						// If it's valid to place the piece, add it to the board
						addPiece(field, pieceToPlace, pentID, x, y);

						// Display the field with the current placement
						ui.setState(field);

						// Recurse to the next pentomino
						if (bruteForceSearch(field, index + 1)) {
							return true;
						}
						// Remove the pentomino from the board (backtrack)
						removePiece(field, pieceToPlace, x, y);
					}
				}
			}
		}
		return false;
	}

	/**
	 * Brute force search algorithm. It checks all the possible combinations of pentominoes and positions to find a solution.
	 * @param field a matrix representing the board to be fulfilled with pentominoes
	 * @param index places pentomino number "index" on the board (if possible)
	 * Calls itself recursively with the next index (field, 1), (field, 2), etc. - until done
	 */

	private static boolean backTrackSearch(int[][] field, int index) {
		if (index == input.length) {
			// All pentominoes placed, a solution is found
			ui.setState(field);
			return true;
		}

		// Retrieve the current pentomino character from the input array
		char piece = input[index];
		int pentID = characterToID(piece);

		// Iterate through *all possible* mutations of the current pentomino piece
		for (int mutation = 0; mutation < PentominoDatabase.data[pentID].length; mutation++) {
			int[][] pieceToPlace = PentominoDatabase.data[pentID][mutation];

			// Place the current pentomino piece at *all possible* positions on the grid
			for (int x = 0; x <= horizontalGridSize - pieceToPlace.length; x++) {
				for (int y = 0; y <= verticalGridSize - pieceToPlace[0].length; y++) {

					if (canPlacePiece(field, pieceToPlace, x, y)) {
						// If it's valid to place the piece, add it to the board
						addPiece(field, pieceToPlace, pentID, x, y);

						// Check if the piece introduces a single hole
						if (introducesHoles(field, pieceToPlace, x, y)) {
							removePiece(field, pieceToPlace, x, y);
							continue;
						}

						// Check if the piece introduces larger, but not refillable spaces
						if (introducesNotRefillableSpaces(field, pieceToPlace, x, y)) {
							removePiece(field, pieceToPlace, x, y);
							continue;
						}

						// Display the field with the current placement
						ui.setState(field);

						// Recurse to the next pentomino
						if (backTrackSearch(field, index + 1)) {
							return true;
						}
						// Remove the pentomino from the board (backtrack)
						removePiece(field, pieceToPlace, x, y);
					}
				}
			}
		}
		return false;
	}

	// Check if the piece introduces holes (which invalidates the solution) - assumes that the piece is already placed on the board!
	public static boolean introducesHoles(int[][] field, int[][] piece, int x, int y) {
		// -1 and +1 because we need to check the surrounding squares as well
		for (int i = -1; i < piece.length + 1; i++) {
			for (int j = -1; j < piece[0].length + 1; j++) {
				// The new values of X and Y (relative to the field) that will be filled by the piece
				int newX = x + i;
				int newY = y + j;

				if (!isValidPosition(newX, newY)) {
					continue;
				}

				if (field[newX][newY] != EMPTY_SPACE_FIELD) {
					// This is not an empty space, so it can't be enclosed
					continue;
				}

				// Check if enclosed by other pieces
				if (isEnclosed(field, newX, newY)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Check if the piece introduces larger, but not refillable spaces (which invalidates the solution) - assumes that
	 * the piece is already placed on the board!
	 * @param field the board
	 * @param piece the piece to be placed
	 * @param x x position of the piece
	 * @param y y position of the piece
	 * @return true if the piece introduces larger, but not refillable spaces, false otherwise
	 */
	public static boolean introducesNotRefillableSpaces(int[][] field, int[][] piece, int x, int y) {
		// created a visited array to keep track of which squares we have already visited
		int[][] visited = new int[horizontalGridSize][verticalGridSize];
		cleanBoard(visited);

		for (int i = -1; i < piece.length + 1; i++) {
			for (int j = -1; j < piece[0]. length + 1; j++) {
				int newX = x + i;
				int newY = y + j;

				if (!isValidPosition(newX, newY)) {
					continue;
				}

				if (field[newX][newY] != EMPTY_SPACE_FIELD) {
					// This is not an empty space, so it can't be enclosed
					continue;
				}

				if (visited[newX][newY] == 1) {
					// We have already visited this square
					continue;
				}

				int connectedSpaces = countConnectedSpaces(field, visited, newX, newY);
				if (connectedSpaces % 5 != 0) {
					return true;
				}
			}
		}
		return false;
	}

	public static int countConnectedSpaces(int[][] field, int[][] visited, int x, int y) {
		int connectedSpaces = 0;

		if (!isValidPosition(x, y)) {
			return connectedSpaces;
		}

		if (visited[x][y] == 1) {
			// We have already visited this square
			return connectedSpaces;
		}
		visited[x][y] = 1;

		if (field[x][y] != EMPTY_SPACE_FIELD) {
			// This is not an empty space, so it can't be enclosed
			return connectedSpaces;
		}

		return (1 +
				countConnectedSpaces(field, visited, x - 1, y) +
				countConnectedSpaces(field, visited, x + 1, y) +
				countConnectedSpaces(field, visited, x, y - 1) +
				countConnectedSpaces(field, visited, x, y + 1));
	}

	/**
	 * Check if the location x, y is enclosed by other pieces (i.e. it's a hole which can't be filled by another pentomino)
	 * @param field the board
	 * @param x x position
	 * @param y y position
	 * @return true if the position is enclosed, false otherwise
	 */
	public static boolean isEnclosed(int[][] field, int x, int y) {
		// Above
		if (isValidEmpty(field, x, y - 1)) {
			return false;
		}

		// Below
		if (isValidEmpty(field, x, y + 1)) {
			return false;
		}

		// Left
		if (isValidEmpty(field, x - 1, y)) {
			return false;
		}

		// Right
        return !isValidEmpty(field, x + 1, y);
    }

	/**
	 * Helper function to check if a position is valid (not out of bounds)
	 * @param x x position
	 * @param y y position
	 * @return true if the position is valid, false otherwise
	 */
	public static boolean isValidPosition(int x, int y) {
		return x >= 0 && x < horizontalGridSize && y >= 0 && y < verticalGridSize;
	}

	/**
	 * Helper function to check if a position is valid and empty (not out of bounds and not occupied by another piece)
	 * @param field the board
	 * @param x x position
	 * @param y y position
	 * @return true if the position is valid and empty, false otherwise
	 */
	public static boolean isValidEmpty(int[][] field, int x, int y) {
		// invalid position - border, consider it
		return isValidPosition(x, y) && field[x][y] == EMPTY_SPACE_FIELD;
	}
}
